<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use App\Models\UserSubscription;
use Illuminate\Container\Attributes\Log;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\Log as FacadesLog;
use Illuminate\Support\Facades\DB;

class StripeController extends Controller
{
    public function createCheckoutSession(Request $request)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));

        // Map incoming "price" to Stripe Price ID
        // $priceMap = [
            // 5  => env('STRIPE_PRICE_ID_5'),
            // 10 => env('STRIPE_PRICE_ID_10'),
            // 20 => env('STRIPE_PRICE_ID_20'),
        // ];

        // $price = $request->price ?? 5; // default to $5 if not passed
        // $priceId = $priceMap[$price] ?? env('STRIPE_PRICE_ID_5');

        $priceId = env('STRIPE_PRICE_ID_20');
        $user = $request->user();

        $session = Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price' => $priceId,
                'quantity' => 1,
            ]],
            'mode' => 'subscription',
            'success_url' => 'http://localhost/testwordpress/wp-admin/admin.php?page=ai-smart-search',
            'cancel_url' => 'http://localhost/testwordpress/cancel',
            'customer_email' => $user->email,
            'metadata' => [
                'user_id' => $user->id,
            ],
        ]);

        return response()->json([
            'url' => $session->url
        ]);
    }

    public function webhook(Request $request)
    {
        $payload = $request->getContent();
        $sigHeader = $request->server('HTTP_STRIPE_SIGNATURE');
        $secret = env('STRIPE_WEBHOOK_SECRET');

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $sigHeader,
                $secret
            );
        } catch (\Exception $e) {
            FacadesLog::error('Stripe webhook signature failed', [
                'error' => $e->getMessage()
            ]);
            return response('Invalid signature', 400);
        }
        FacadesLog::info($event->type);
        /* =====================================================
        | STEP 1 — CHECKOUT SESSION COMPLETED (SOURCE OF TRUTH)
        ===================================================== */
        if ($event->type === 'checkout.session.completed') {

            $session = $event->data->object;

            if ($session->mode !== 'subscription') {
                return response('Not subscription checkout', 200);
            }

            $userId = $session->metadata->user_id ?? null;

            if (!$userId) {
                FacadesLog::error('Missing user_id in checkout session', [
                    'session_id' => $session->id,
                ]);
                return response('Missing user_id', 200);
            }

            UserSubscription::updateOrCreate(
                ['user_id' => $userId],
                [
                    'email'           => $session->customer_email,
                    'token'           => $session->customer,        // Stripe customer ID
                    'subscription_id' => $session->subscription,    // sub_***
                    'status'          => 'pending',
                ]
            );

            FacadesLog::info('Checkout completed — subscription pending', [
                'user_id' => $userId,
                'customer_id' => $session->customer,
                'subscription_id' => $session->subscription,
            ]);
        }

        if ($event->type === 'invoice.payment_succeeded') {

            $invoice = $event->data->object;

            $subscriptionId = $invoice->subscription ?? $invoice->parent->subscription_details->subscription ?? null;

            if (!$subscriptionId) {
                FacadesLog::info('Invoice without subscription ignored', [
                    'invoice_id' => $invoice->id
                ]);
                return response('No subscription', 200);
            }

            $updated = UserSubscription::where('subscription_id', $invoice->subscription)
                ->orWhere('token', $invoice->customer) // 🔥 IMPORTANT
                ->update([
                    'subscription_id' => $subscriptionId,
                    'status' => 'active',
                    'expires_at' => \Carbon\Carbon::createFromTimestamp(
                        $invoice->lines->data[0]->period->end
                    ),
                ]);

            FacadesLog::info('Invoice payment processed', [
                'invoice_id' => $invoice->id,
                'subscription_id' => $invoice->subscription,
                'customer_id' => $invoice->customer,
                'rows_updated' => $updated, // 👈 THIS WILL TELL THE TRUTH
            ]);
        }

        return response('Webhook handled', 200);

    }


    public function checkSubscription(Request $request)
    {
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'Unauthenticated'], 401);
        }

        $subscription = $user->subscription;

        $subscriptionActive = $subscription
            && $subscription->status === 'active'
            && (!$subscription->expires_at || $subscription->expires_at > now());

        return response()->json([
            'subscription_active' => $subscriptionActive,
            'expires_at' => $subscription?->expires_at,
        ]);
    }


    // public function getSubscriptionLogs()
    // {
    //     return DB::table('user_subscriptions')
    //         ->orderBy('id', 'desc')
    //         ->get();
    // }
}
